/**
* \file        Uart_SC16IS740.cpp
* \brief       External Uart SC16IS740/750/760 Hardware class Definitions
* \author      Andrew Salmon
* \date        07/13/2009
*/
#include <string.h>
#include "Uart_SC16IS740.hpp"
#include "dm_os.h"
#include "dm_spi.h"
#include "dm_console.h"
#include "console.h"
#include "dm_hardware.h"
#include "dm_device.h"
#include "dm_errors.h"
#include "dm_field_debug.h"

static Uart_SC16IS740 *g_pExtUart = 0;//NULL

/**
 * \author    Andrew Salmon
 * \brief     Callback function for the external device read task;  This is a C wrapper
 *            to access the non-static method of the SerialPort class; once called,
 *            the method never returns
 * \date      04/10/2010
 * \param     pSerialportObj - pointer to the Serialport object
 * \return    void
 * \retval    none
**/
void ProcessDeviceTaskCallback(UINT32 pObj)
{
  ((Uart_SC16IS740*)pObj)->ProcessUartTask();
}
/**
* \author        Andrew Salmon
* \brief         constructor for SC16IS740 hardware access class
* \date          07/13/2009
*
* \param         pHwCfg - pointer to hardware configuration
*
**/
Uart_SC16IS740::Uart_SC16IS740( const SC16IS740Cfg *pHwCfg):
    m_HwCfg(NULL),
    m_pHwAccess(NULL),
    m_FramingErrorCnt(0),
    m_OverrunErrorCnt(0),
    m_ParityErrorCnt(0),
    m_FIFOErrorCnt(0),
    m_EventGroup(0),
    mQueueCount(0),
    m_HandshakeCTS(0),
    m_HandshakeRTS(0)
{
    if(!pHwCfg)
       return;

    m_HwCfg           = pHwCfg;

    //Perform a hardware reset
    HwReset();

    if(m_HwCfg->bBusSelect == SCI_BUS_I2C)
    {
        m_pHwAccess = new SC16IS740Hw_I2C( m_HwCfg );
    }
    else
    {
        m_pHwAccess = new SC16IS740Hw_SPI( m_HwCfg );
    }

    //Initialize the falling edge interrupts
    InitializeEdgeInt(false);
    DisableEdgeInt();//will enable when Serialport Init

    if(m_HwCfg->OwnTask)
    {
       OsCreateNamedAdvTask(ProcessDeviceTaskCallback, OsGetDefaultStackVarCnt(),(UINT32*)this,  pHwCfg->taskPrio, "EXTR" );
    }
}
/**
 * \author    Andrew Salmon
 * \brief     creates the UART SC16I6740 class
 * \date      07/20/09
 * \param     pHwCfg -pointer to the configuration table
 *
**/
void CreateSC16IS740UartClass(const SC16IS740Cfg *pHwCfg)
{
    g_pExtUart = new Uart_SC16IS740(pHwCfg);
}

/**
 *  \author     Andrew Salmon
 *
 *  \date       07/27/09 11:09:AM
 *
 *  \brief      retreive event from task queue
 *
 * \param       event pointer to hold received data
 *
 */
void UartSC16IS740ServiceUart(void)
{
   if(g_pExtUart)
   {
      g_pExtUart->ServiceDevice();
   }
}
void Uart_SC16IS740::ServiceDevice( void )
{
   UINT32 event;
   UINT32 waitTime = 0;

   // SerialPort Driver uses shared task and cannot be blocked
   if(m_HwCfg->OwnTask)
   {
      waitTime = SC16IS740_SERVICE_QUEUE_PERIOD;
   }

   if(OsDequeueWait(m_EventGroup, (void *) &event, waitTime , /*UINT32 no_consume*/ false) == 0)
    {
       //read the device
       ProcessIRQEvent();

       //check interrupt still active
       while(InterruptActiveStatus())
       {
          ProcessIRQEvent();
       }
    }
}
/**
 *  \author     Andrew Salmon
 *
 *  \date       04/15/10 11:09:AM
 *
 *  \brief      create msg queue for device
 *
 */
void UartSC16IS740CreateMsgQueue(void)
{
   if(g_pExtUart)
   {
      g_pExtUart->CreateMsgQueue();
   }
}
void UartSC16IS740DestroyMsgQueue(void)
{
   if(g_pExtUart)
   {
      g_pExtUart->DestroyMsgQueue();
   }
}
void Uart_SC16IS740::CreateMsgQueue(void)
{
   // Create EventGroups
   m_EventGroup = OsQueueCreate(m_HwCfg->QueueSize, sizeof(UINT8) );

   if(m_EventGroup == 0 )
   {
      DmSystemError(DM_ERROR_LEVEL_FATAL, DM_ERROR_SUBSYS_EXT_UART, ERR_EXT_UART_TASK_NO_EVENT, 0);
   }
}
void Uart_SC16IS740::DestroyMsgQueue(void)
{
   if(m_EventGroup)
   {
      OsQueueDelete(m_EventGroup);
      m_EventGroup = 0;
   }
}
void UartSC16IS740DisableHardware(void)
{
   if(g_pExtUart)
        g_pExtUart->DisableHardware();
}
/**
 *  \author     Andrew Salmon
 *
 *  \date       07/27/09 11:09:AM
 *
 *  \brief      retreive event from task queue
 *
 * \param       event pointer to hold received data
 *
 */
void Uart_SC16IS740::ProcessUartTask( void )
{
   CreateMsgQueue();

   while (1)
   {
       ServiceDevice();
   }

}
/**
 * \author      Andrew Salmon
 *
 * \date        07/16/2009
 *
 * \brief       Transmit data bytes
 *
 * \param       data pointer to serial data to transmit
 * \param       count number of bytes
 * \param       inst uart instance
 *
 */
void Uart_SC16IS740::ConsoleTx(UINT8 *data, UINT32 count, UINT32 inst)
{
    while(count--)
    {
        UART_SendData(*data++);
        if(!InIsr)
        {
            // temp only - change this to be interrupt driven
            HwDelayMsec(1);
        }
    }
}
/**
* \author         Andrew salmon
* \brief          Debug hardware specific registers and print the debug messages
* \date           07/16/09
*
*/
void Uart_SC16IS740::TestTxRx(void)
{
#ifdef DEBUG_SC16IS740_UART

   UINT8 bData, bData1;
   //enable loopback
   EnableLoopback( true );
   //is loobback mode entered
   bData = m_pHwAccess->ReadRegister(UART_MCR_ADDR);

   if(bData&UMCR_LB)
   {
      DmConsolePrintf("\r\nLoopback Enabled\r");
      //save the configuration
      bData1 = m_pHwAccess->ReadRegister(UART_IER_ADDR);
      //disable rx interrupt
      m_pHwAccess->ClearRegisterBits(UART_IER_ADDR, UIER_RDAIE);
      //write a byte
      m_pHwAccess->WriteRegister(UART_THR_ADDR, 0xA5);

      //read back byte
      bData = m_pHwAccess->ReadRegister(UART_RHR_ADDR);
      if(bData == 0xA5)
      {
         DmConsolePrintf("\r\nTX/RX OK\r");
      }
      else
      {
         DmConsolePrintf("\r\nTX/RX Failed\r");
      }
      //write a byte
      m_pHwAccess->WriteRegister(UART_IER_ADDR, bData1);
   }
   else
   {
      DmConsolePrintf("\r\nLoopback NOT Enabled\r");
   }

   //disable loopback
   EnableLoopback( false );
#endif// DEBUG_SC16IS740_UART
}
/**
* \author         Andrew salmon
* \brief          Debug hardware specific registers and print the debug messages
* \date           07/16/09
*
*/
void Uart_SC16IS740::PrintDebug(void)
{
    UINT8 bData;

    //Interrupt Enable register
    bData = m_pHwAccess->ReadRegister(UART_IER_ADDR);

#ifdef DEBUG_SC16IS740_UART
    //Print the status bits
    DmConsolePrintf("\r\nInterrupt Enable register\r");
    DmConsolePrintf("RX enable: %d\r", ((bData & UIER_RDAIE) ? 1 : 0) );
    DmConsolePrintf("TX eneble: %d\r", ((bData & UIER_THREIE) ? 1 : 0) );
    DmConsolePrintf("RX Line Status: %d\r", ((bData & UIER_RXLSIE) ? 1 : 0) );
    DmConsolePrintf("Modem status: %d\r", ((bData & UIER_MSIE) ? 1 : 0) );
    DmConsolePrintf("SLEEP Mode: %d\r", ((bData & UIER_SM) ? 1 : 0) );
    DmConsolePrintf("XOFF: %d\r", ((bData & UIER_XOFFIE) ? 1 : 0) );
    DmConsolePrintf("RTS: %d\r", ((bData & UIER_RTSIE) ? 1 : 0) );
    DmConsolePrintf("CTS: %d\r", ((bData & UIER_CTSIE) ? 1 : 0) );
#else
    DmConsolePrintf("IER: 0x%x\r", bData );
#endif

    //Interrupt Identification register
    bData = m_pHwAccess->ReadRegister(UART_IIR_ADDR);

#ifdef DEBUG_SC16IS740_UART
    //Print the status bits
    DmConsolePrintf("\r\nInterrupt Identification register\r");
    DmConsolePrintf("Interrupt Status: %d\r", ((bData & UIIR_IP) ? 1 : 0) );
    DmConsolePrintf("TX buffer empty: %d\r", ((bData & UIIR_THRE) ? 1 : 0) );
    DmConsolePrintf("RX data available: %d\r", ((bData & UIIR_RDA) ? 1 : 0) );
    DmConsolePrintf("Character Timeout: %d\r", ((bData & UIIR_CTI) ? 1 : 0) );
    DmConsolePrintf("Xoff Enabled: %d\r", ((bData & UIIR_SC) ? 1 : 0) );
    DmConsolePrintf("CTS/RTS Changed state: %d\r", ((bData & UIIR_CTSRTS) ? 1 : 0) );
    DmConsolePrintf("FIFO Enable: %d\r", ((bData & 0x40) ? 1 : 0) );
    DmConsolePrintf("FIFO Enable: %d\r", ((bData & 0x80) ? 1 : 0) );
#else
    DmConsolePrintf("IIR: 0x%x\r", bData );
#endif

    //Line Control register
    bData = m_pHwAccess->ReadRegister(UART_LCR_ADDR);

#ifdef DEBUG_SC16IS740_UART
    //Print the status bits
    DmConsolePrintf("\r\nLine Control register\r");
    DmConsolePrintf("Word Len bit 0: %d\r", ((bData & ULCR_WLS0) ? 1 : 0) );
    DmConsolePrintf("Word Len bit 1: %d\r", ((bData & ULCR_WLS1) ? 1 : 0) );
    DmConsolePrintf("Stop bit: %d\r", ((bData & ULCR_SBS) ? 1 : 0) );
    DmConsolePrintf("Parity Enabled: %d\r", ((bData & ULCR_PE) ? 1 : 0) );
    DmConsolePrintf("Parity even: %d\r", ((bData & ULCR_PS0) ? 1 : 0) );
    DmConsolePrintf("Parity set: %d\r", ((bData & ULCR_PS1) ? 1 : 0) );
    DmConsolePrintf("Break set: %d\r", ((bData & ULCR_BC) ? 1 : 0) );
    DmConsolePrintf("Divisor latch Enable: %d\r", ((bData & ULCR_DLAB) ? 1 : 0) );
#else
    DmConsolePrintf("LCR: 0x%x\r", bData );
#endif

    //Modem Control register
    bData = m_pHwAccess->ReadRegister(UART_MCR_ADDR);

#ifdef DEBUG_SC16IS740_UART
    //Print the status bits
    DmConsolePrintf("\r\nModem Control register\r");
    DmConsolePrintf("DTR: %d\r", ((bData & UMCR_DTR) ? 1 : 0) );
    DmConsolePrintf("RTS: %d\r", ((bData & UMCR_RTS) ? 1 : 0) );
    DmConsolePrintf("TCR and TLR enable: %d\r", ((bData & UMCR_TCR) ? 1 : 0) );
    DmConsolePrintf("Loopback: %d\r", ((bData & UMCR_LB) ? 1 : 0) );
    DmConsolePrintf("XON any: %d\r", ((bData & UMCR_XONA) ? 1 : 0) );
    DmConsolePrintf("IrDA: %d\r", ((bData & UMCR_IrDA) ? 1 : 0) );
    DmConsolePrintf("Clock Divisor: %d\r", ((bData & UMCR_CD) ? 1 : 0) );
#else
    DmConsolePrintf("MCR: 0x%x\r", bData );
#endif

#ifdef DEBUG_SC16IS740_UART
     EnableEnhancedFunctions( true);
     //Modem Control register
    bData = m_pHwAccess->ReadRegister(UART_TCR_ADDR);
    DmConsolePrintf("TCR : %d\r", bData);
    bData = m_pHwAccess->ReadRegister(UART_TLR_ADDR);
    DmConsolePrintf("TLR : %d\r", bData);

    EnableEnhancedFunctions( false);

    //Modem Control register
    bData = m_pHwAccess->ReadRegister(UART_TXLVL_ADDR);
    DmConsolePrintf("TXLVL : %d\r", bData);
    bData = m_pHwAccess->ReadRegister(UART_RXLVL_ADDR);
    DmConsolePrintf("RXLVL : %d\r", bData);
#endif

    //Extra Features Control register
    bData = m_pHwAccess->ReadRegister(UART_EFCR_ADDR);

#ifdef DEBUG_SC16IS740_UART
    //Print the status bits
    DmConsolePrintf("\r\nExtra Feature Control register\r");
    DmConsolePrintf("9-bit Mode: %d\r", ((bData & UEFCR_NBIT) ? 1 : 0) );
    DmConsolePrintf("rx disable: %d\r", ((bData & UEFCR_RXE) ? 1 : 0) );
    DmConsolePrintf("tx disable: %d\r", ((bData & UEFCR_TXE) ? 1 : 0) );
    DmConsolePrintf("auto RTS dir: %d\r", ((bData & UEFCR_ARTSD) ? 1 : 0) );
    DmConsolePrintf("auto RTS output inv: %d\r", ((bData & UEFCR_ARTSO) ? 1 : 0) );
    DmConsolePrintf("IrDA: %d\r", ((bData & UEFCR_IrDA) ? 1 : 0) );
#else
    DmConsolePrintf("EFCR: 0x%x\r", bData );
#endif

    //Divisor Latch registers
    //set LCR to access special register
    m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, ULCR_DLAB);
    //read the register
    bData = m_pHwAccess->ReadRegister(UART_DLL_ADDR);
    //print the register
    DmConsolePrintf("\r\nDivisor Latch LSB : 0x%x\r", bData);
    //read the MSB register
    bData = m_pHwAccess->ReadRegister(UART_DLH_ADDR);
    //print the register
    DmConsolePrintf("\r\nDivisor Latch MSB : 0x%x\r", bData);
    //Clear access to special register
    m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, ULCR_DLAB);

    //FIFO Control register
    bData = m_pHwAccess->GetFCR_Register();

#ifdef DEBUG_SC16IS740_UART
    //Print the status bits
    DmConsolePrintf("\r\nFIFO Control register\r");
    DmConsolePrintf("FIFO Enabled: %d\r", ((bData & UFCR_FCRFE) ? 1 : 0) );
    DmConsolePrintf("Rx FIFO reset: %d\r", ((bData & UFCR_RFR) ? 1 : 0) );
    DmConsolePrintf("Tx FIFO reste: %d\r", ((bData & UFCR_TFR) ? 1 : 0) );
    DmConsolePrintf("Tx Trig Level LSB: %d\r", ((bData & UFCR_TTLS0) ? 1 : 0) );
    DmConsolePrintf("Tx Trig Level MSB: %d\r", ((bData & UFCR_TTLS1) ? 1 : 0) );
    DmConsolePrintf("Rx Trig Level LSB: %d\r", ((bData & UFCR_RTLS0) ? 1 : 0) );
    DmConsolePrintf("Rx Trig Level MSB: %d\r", ((bData & UFCR_RTLS1) ? 1 : 0) );
#else
    DmConsolePrintf("FCR: 0x%x\r", bData);
#endif// DEBUG_SC16IS740_UART
}
/**
 * \author      Andrew Salmon
 *
 * \date        09/18/2008
 *
 * \return      UINT8
 *
 * \retval      register value
 *
 * \brief       Reads state of UART CTS pin
 *
 */
UINT8 Uart_SC16IS740::GetCTS(void)
{
   // read CTS bit, active high complement of /CTS pin
   return ((m_pHwAccess->ReadRegister(UART_MSR_ADDR)&UMSR_CTS) ? false:true);
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/15/2009
 *
 *
 * \brief       Change UART RTS state
 *
 * \param       state - Bit_SET (asserts)/Bit_Reset (deasserts)
 */
void Uart_SC16IS740::AssertRTS( UINT8 state)
{
   // has no effect if auto is enabled
   if(m_HwCfg->bAutoRTS && m_HandshakeRTS)
     return;

   if(state)
      m_pHwAccess->ClearRegisterBits(UART_MCR_ADDR, UMCR_RTS);//forces /RTS inactive high
   else
      m_pHwAccess->SetRegisterBits(UART_MCR_ADDR, UMCR_RTS);
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/15/2009
 *
 * \brief       Change UART loopback state
 *
 * \param       state - Bit_SET /Bit_Reset
 */
void Uart_SC16IS740::EnableLoopback( UINT8 state)
{
   if( state )
      m_pHwAccess->SetRegisterBits(UART_MCR_ADDR, UMCR_LB);
   else
      m_pHwAccess->ClearRegisterBits(UART_MCR_ADDR, UMCR_LB);
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/15/2009
 *
 * \brief       Change UART Auto CTS state
 *
 * \param       state - Bit_SET /Bit_Reset
 */
void Uart_SC16IS740::WriteEnahancedFeatureRegister( UINT8 state, UINT8 bDataBits)
{
   UINT8 data;

   //save LCR
   data = m_pHwAccess->ReadRegister(UART_LCR_ADDR);;

   //set LCR to access special register
   m_pHwAccess->WriteRegister(UART_LCR_ADDR, 0xBF);

   if( state )
      m_pHwAccess->SetRegisterBits(UART_EFR_ADDR, bDataBits);
   else
      m_pHwAccess->ClearRegisterBits(UART_EFR_ADDR, bDataBits);

   //clear LCR to access general purpose register
   m_pHwAccess->WriteRegister(UART_LCR_ADDR, data);
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/15/2009
 *
 * \brief       Change UART Auto CTS state
 *
 * \param       state - Bit_SET /Bit_Reset
 */
void Uart_SC16IS740::EnableAutoCTS( UINT8 state)
{
   WriteEnahancedFeatureRegister( state, UEFR_CTS);
   return;
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/15/2009
 *
 * \brief       Change UART Auto RTS state
 *
 * \param       state - Bit_SET /Bit_Reset
 */
void Uart_SC16IS740::EnableAutoRTS( UINT8 state)
{
   WriteEnahancedFeatureRegister( state, UEFR_RTS);
   return;
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/15/2009
 *
 * \brief       Change enhanced function state
 *
 * \param       state - Bit_SET /Bit_Reset
 */
void Uart_SC16IS740::EnableEnhancedFunctions( UINT8 state)
{
   WriteEnahancedFeatureRegister( state, UEFR_EN);
   return;
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/15/2009
 *
 * \brief       Set baud rate divisor latch
 *
 * \param       lsb divisor latch word LSB
 * \param       msb divisor latch word MSB
 */
void Uart_SC16IS740::SetBaudDivisorLatchWord( UINT8 lsb, UINT8 msb)
{
   //set LCR to access special register
   m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, ULCR_DLAB);

   m_pHwAccess->WriteRegister(UART_DLL_ADDR, lsb);
   m_pHwAccess->WriteRegister(UART_DLH_ADDR, msb);

   m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, ULCR_DLAB);

}
/**
 * \author    Andrew Salmon
 * \brief     Send data to transmitter.
 * \date      07/14/2009
 * \param     Data byte to transmit
 *
**/
void Uart_SC16IS740::UART_SendData( UINT8 Data)
{
   /* Transmit one byte */
   m_pHwAccess->WriteRegister(UART_THR_ADDR, Data);
   return;
}
/**
 * \author      Andrew Salmon
 *
 * \date        09/17/2008
 *
 * \brief       Configure all programable parameters of UART
 *
 * \param       inst - which uart
 * \param       bps - baud rate
 * \param       stopBits - number of stop bits
 * \param       parity - aprity
 * \param       wlenght - word length
 * \param       hwcontrol - hardware flow control
 *
 */
void Uart_SC16IS740::HwConfig(UINT32 inst, UINT32 bps, UINT32 stopBits, UINT32 parity, UINT32 wlenght, UINT32 hwcontrol)
{
   UINT8 lsb, msb, prs;

   //disable the device interrupt
   DisableEdgeInt();
   //Perform a hardware reset
   HwReset();

   SetBaudrate( bps, (UINT8 *)&lsb, (UINT8 *)&msb, (UINT8 *)&prs);

   if( prs > 1)
   {
      //dividex4 clock selected
      m_pHwAccess->SetRegisterBits(UART_MCR_ADDR, UMCR_CD);
   }
   // set baud rate divisor
   SetBaudDivisorLatchWord( lsb, msb);

   switch(stopBits)
   {
   default:
   case 1:
      m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, ULCR_SBS);
      break;

   case 2:
      m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, ULCR_SBS);
      break;
    }
   //set parity
   switch(parity)
   {
   case SCI_UART_Parity_No:
   default:
     m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, (ULCR_PE|ULCR_PS0|ULCR_PS1));
     break;

   case SCI_UART_Parity_Even:
      m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, (ULCR_PE|ULCR_PS0));
      m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, ULCR_PS1);
      break;
   case SCI_UART_Parity_Odd:
      m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, ULCR_PE);
      m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, (ULCR_PS0|ULCR_PS1));
      break;
   case SCI_UART_Parity_OddStick:
      m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, (ULCR_PE|ULCR_PS0|ULCR_PS1));
      break;
   case SCI_UART_Parity_EvenStick:
      m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, (ULCR_PE|ULCR_PS1));
      m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, ULCR_PS0);
      break;
   }
   //set wordlength
   switch(wlenght)
   {
   default:
   case 8:
      m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, (ULCR_WLS0|ULCR_WLS1));
      break;

   case 7:
      m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, ULCR_WLS0);
      m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, ULCR_WLS1);
      break;

   case 6:
      m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, ULCR_WLS1);
      m_pHwAccess->SetRegisterBits(UART_LCR_ADDR, ULCR_WLS0);
      break;

   case 5:
      m_pHwAccess->ClearRegisterBits(UART_LCR_ADDR, (ULCR_WLS0|ULCR_WLS1));
      break;
   }

   if(m_HwCfg->bEnableFIFOs)
   {
      SetRxFIFOTriggerLevel();
      SetTxFIFOTriggerLevel();

      // reset and enable Tx/Rx FIFO
      m_pHwAccess->SetRegisterBits( UART_FCR_ADDR, UFCR_RFR | UFCR_TFR | UFCR_FCRFE );
   }
   else
   {
      // reset and disable Tx/Rx FIFO
      m_pHwAccess->SetRegisterBits( UART_FCR_ADDR, UFCR_RFR | UFCR_TFR );
      m_pHwAccess->ClearRegisterBits( UART_FCR_ADDR, UFCR_FCRFE );
   }

   if(hwcontrol & SCI_UART_COMSPEC_CTS)
   {
       m_HandshakeCTS = 1;
   }
   else
   {
       m_HandshakeCTS = 0;
   }


   if(hwcontrol & SCI_UART_COMSPEC_RTS)
   {
       m_HandshakeRTS =1;
   }
   else
   {
      m_HandshakeRTS = 0;
   }

   if( (m_HandshakeRTS && m_HwCfg->bAutoRTS) || (m_HandshakeCTS && m_HwCfg->bAutoCTS ))
   {
       SetRxFIFOHaultResumeTriggerLevel();

       if( m_HwCfg->bAutoCTS )
       {
           EnableAutoCTS(1);
       }

       if( m_HwCfg->bAutoRTS )
       {
           EnableAutoRTS(1);
       }
   }

   // Enable UART interrupts
   m_pHwAccess->SetRegisterBits(UART_IER_ADDR, UIER_RDAIE|UIER_RXLSIE);
   //enable the falling edge interrupt
   EnableEdgeInt(false);

}
/**
 * \author    Andrew Salmon
 * \brief     Setup Rx FIFO trigger levels
 * \date      07/24/2009
 *
**/
void Uart_SC16IS740::SetRxFIFOTriggerLevel(void)
{
   if(m_HwCfg->bRxTriggerLevel == RX_FIFO_LEVEL_0)//use default
      return;

   if(m_HwCfg->bRxTriggerLevel <= RX_FIFO_LEVEL_8)
   {
      //make it 8 spaces
      m_pHwAccess->ClearRegisterBits(UART_FCR_ADDR, (UFCR_RTLS1|UFCR_RTLS0));
   }
   else if (m_HwCfg->bRxTriggerLevel <= RX_FIFO_LEVEL_16)
   {
      //make it 16 spaces
      m_pHwAccess->ClearRegisterBits(UART_FCR_ADDR, UFCR_RTLS1);
      m_pHwAccess->SetRegisterBits(UART_FCR_ADDR, UFCR_RTLS0);
   }
   else if (m_HwCfg->bRxTriggerLevel <= RX_FIFO_LEVEL_56)
   {
      //make it 56 spaces
      m_pHwAccess->ClearRegisterBits(UART_FCR_ADDR, UFCR_RTLS0);
      m_pHwAccess->SetRegisterBits(UART_FCR_ADDR, UFCR_RTLS1);
   }
   else// if (m_HwCfg->bRxTriggerLevel <= RX_FIFO_LEVEL_60)
   {
      //make it 60 spaces
      m_pHwAccess->SetRegisterBits(UART_FCR_ADDR, (UFCR_RTLS0 | UFCR_RTLS1));
   }
}
/**
 * \author    Andrew Salmon
 * \brief     Setup Rx FIFO resume/halt trigger levels
 * \date      01/04/2011
 *
**/
void Uart_SC16IS740::SetRxFIFOHaultResumeTriggerLevel(void)
{
   UINT8 hualtRx = 0, resumeRx = 0;
#ifdef DEBUG_SC16IS740_UART
   UINT8 bData;
#endif

   if((m_HwCfg->bRxFifoHalt >= m_HwCfg->bRxFifoResume) || (m_HwCfg->bRxFifoHalt > RX_HALT_RESUME_LEVEL_MAX ) || (m_HwCfg->bRxFifoResume > RX_HALT_RESUME_LEVEL_MAX ))
   {
      if((m_HwCfg->bRxFifoHalt > RX_HALT_RESUME_LEVEL_MAX ) || (m_HwCfg->bRxFifoResume > RX_HALT_RESUME_LEVEL_MAX ))
      {
          hualtRx  = ((RX_HALT_RESUME_LEVEL_MAX*90)/100);
          resumeRx = ((RX_HALT_RESUME_LEVEL_MAX*60)/100);
      }
      else
      {
          // if incorrect setting asume 60/90%
          hualtRx  = ((m_HwCfg->bRxTriggerLevel*90)/100);
          resumeRx = ((m_HwCfg->bRxTriggerLevel*60)/100);
      }
   }
   else
   {
      hualtRx  = m_HwCfg->bRxFifoHalt;
      resumeRx = m_HwCfg->bRxFifoResume;

   }

   // check if they fall in the same range
   if(((hualtRx <= 15) && (resumeRx <= 15)) ||
      ((hualtRx <= 30) && (resumeRx <= 30)) || ((hualtRx <= 45) && (resumeRx <= 45)) || ((hualtRx <= RX_HALT_RESUME_LEVEL_MAX) && (resumeRx <= RX_HALT_RESUME_LEVEL_MAX)) )
   {
       // reduce the resume level
       resumeRx = ((resumeRx*60)/100);
   }

   // enable cts/rts trigger level..bug 57459 - enable Enhanced feature before TCR enable in MCR
   EnableEnhancedFunctions( true);
   m_pHwAccess->SetRegisterBits( UART_MCR_ADDR, UMCR_TCR );

   // Configure halt level
   if(hualtRx <= 15)
   {
      //make it 15 char
      m_pHwAccess->ClearRegisterBits(UART_TCR_ADDR, UTCR_RFHTLS3|UTCR_RFHTLS2|UTCR_RFHTLS1);
      m_pHwAccess->SetRegisterBits(UART_TCR_ADDR, UTCR_RFHTLS0);
   }
   else if (hualtRx <= 30)
   {
      //make it 30 char
      m_pHwAccess->ClearRegisterBits(UART_TCR_ADDR, UTCR_RFHTLS3|UTCR_RFHTLS2);
      m_pHwAccess->SetRegisterBits(UART_TCR_ADDR, UTCR_RFHTLS0|UTCR_RFHTLS1);
   }
   else if (hualtRx <= 45)
   {
      //make it 45 char
      m_pHwAccess->ClearRegisterBits(UART_TCR_ADDR, UTCR_RFHTLS3);
      m_pHwAccess->SetRegisterBits(UART_TCR_ADDR, UTCR_RFHTLS0|UTCR_RFHTLS1|UTCR_RFHTLS2);
   }
   else// 60
   {
#ifdef DEBUG_SC16IS740_UART
      bData = m_pHwAccess->ReadRegister(UART_TCR_ADDR);
      DmConsolePrintf("TCR Halt Level Bits Before :0x%x - Bits Set:0x%x\r", bData, UTCR_RFHTLS0|UTCR_RFHTLS1|UTCR_RFHTLS2|UTCR_RFHTLS3);
#endif
      //make it 60 char
      m_pHwAccess->SetRegisterBits(UART_TCR_ADDR, UTCR_RFHTLS0|UTCR_RFHTLS1|UTCR_RFHTLS2|UTCR_RFHTLS3);
#ifdef DEBUG_SC16IS740_UART
      bData = m_pHwAccess->ReadRegister(UART_TCR_ADDR);
      DmConsolePrintf("TCR After Halt Level write : 0x%x\r", bData);
#endif
   }

   // configure resume level
   if(resumeRx <= 15)
   {
      //make it 8 spaces
      m_pHwAccess->ClearRegisterBits(UART_TCR_ADDR, UTCR_RFRTLS3|UTCR_RFRTLS2|UTCR_RFRTLS1);
      m_pHwAccess->SetRegisterBits(UART_TCR_ADDR, UTCR_RFRTLS0);
   }
   else if (resumeRx <= 30)
   {
#ifdef DEBUG_SC16IS740_UART
      bData = m_pHwAccess->ReadRegister(UART_TCR_ADDR);
      DmConsolePrintf("TCR Before Resume Bits Before:0x%x ", bData );
      DmConsolePrintf("- Bits Set:0x%x", UTCR_RFRTLS0|UTCR_RFRTLS1 );
      DmConsolePrintf("- Bits Cleared:0x%x\r", UTCR_RFRTLS3|UTCR_RFRTLS2 );
#endif
      //make it 16 spaces
      m_pHwAccess->ClearRegisterBits(UART_TCR_ADDR, UTCR_RFRTLS3|UTCR_RFRTLS2);
      m_pHwAccess->SetRegisterBits(UART_TCR_ADDR, UTCR_RFRTLS0|UTCR_RFRTLS1);
#ifdef DEBUG_SC16IS740_UART
      bData = m_pHwAccess->ReadRegister(UART_TCR_ADDR);
      DmConsolePrintf("TCR After Resume Level Write: 0x%x\r", bData);
 #endif
   }
   else if (resumeRx <= 45)
   {
      //make it 56 spaces
      m_pHwAccess->ClearRegisterBits(UART_TCR_ADDR, UTCR_RFRTLS3);
      m_pHwAccess->SetRegisterBits(UART_TCR_ADDR, UTCR_RFRTLS0|UTCR_RFRTLS1|UTCR_RFRTLS2);
   }
   else// 60
   {
      //make it 60 spaces
      m_pHwAccess->SetRegisterBits(UART_TCR_ADDR, UTCR_RFRTLS0|UTCR_RFRTLS1|UTCR_RFRTLS2|UTCR_RFRTLS3);
   }

   m_pHwAccess->ClearRegisterBits( UART_MCR_ADDR, UMCR_TCR );
   EnableEnhancedFunctions( false);
}
/**
 * \author    Andrew Salmon
 * \brief     Setup Tx FIFO trigger levels
 * \date      07/24/2009
 *
**/
void Uart_SC16IS740::SetTxFIFOTriggerLevel(void)
{
   if(m_HwCfg->bTxTriggerLevel == TX_FIFO_LEVEL_0)//use default
     return;

   EnableEnhancedFunctions( true);

   if(m_HwCfg->bTxTriggerLevel <= TX_FIFO_LEVEL_8)
   {
      //make it 8 spaces
      m_pHwAccess->ClearRegisterBits(UART_FCR_ADDR, (UFCR_TTLS1|UFCR_TTLS0));
   }
   else if (m_HwCfg->bTxTriggerLevel <= TX_FIFO_LEVEL_16)
   {
      //make it 16 spaces
      m_pHwAccess->ClearRegisterBits(UART_FCR_ADDR, UFCR_TTLS1);
      m_pHwAccess->SetRegisterBits(UART_FCR_ADDR, UFCR_TTLS0);
   }
   else if (m_HwCfg->bTxTriggerLevel <= TX_FIFO_LEVEL_32)
   {
      //make it 56 spaces
      m_pHwAccess->ClearRegisterBits(UART_FCR_ADDR, UFCR_TTLS0);
      m_pHwAccess->SetRegisterBits(UART_FCR_ADDR, UFCR_TTLS1);
   }
   else// if (m_HwCfg->bRxTriggerLevel <= TX_FIFO_LEVEL_56)
   {
      //make it 60 spaces
      m_pHwAccess->SetRegisterBits(UART_FCR_ADDR, (UFCR_TTLS0 | UFCR_TTLS1));
   }
   EnableEnhancedFunctions( false);
}
/**
 * \author    Andrew Salmon
 * \brief     This function sends the first byte then init the tx interrupt to send the rest.
 * \date      07/14/2009
 * \param       uartNum uart instance
 * \param       data byte to transmit
 *
**/
void Uart_SC16IS740::HwUartTx(UINT32 uartNum, UINT8 data)
{
   UART_SendData(data);
   //enable the tx interrupt
   m_pHwAccess->SetRegisterBits(UART_IER_ADDR, UIER_THREIE);
}
/**
 * \author    Andrew Salmon
 * \brief     This function process irq events from the serial task
 * \date      07/14/2009
 *
**/
void Uart_SC16IS740::ProcessIRQEvent( void )
{
   UINT8  IntState, RxStatus, RxData=0, FifoLevel, rxReady=0;
   UINT32 isrFlags;

   IntState = m_pHwAccess->ReadRegister(UART_IIR_ADDR); //interrupt identification register address

   if(IntState & UIIR_IP)
     return;//no interrupt pending

   switch ( IntState & UIIR_IID )
   {
   case UIIR_RLS:                  // First priority, receive line status interrupt code
   case UIIR_RDA:                  // Second priority, Receive data available interrupt code

      RxStatus = m_pHwAccess->ReadRegister(UART_LSR_ADDR);
      RxData   = m_pHwAccess->ReadRegister(UART_RHR_ADDR);//receive holding register address

      if(RxStatus & ULSR_FE)
         m_FramingErrorCnt++;
      if(RxStatus & ULSR_OE)
         m_OverrunErrorCnt++;
      if(RxStatus & ULSR_PE)
         m_ParityErrorCnt++;
      if(RxStatus & ULSR_RXFE)
         m_FIFOErrorCnt++;

      if( IsDmFieldDebugIndexActive(DM_FIELD_DM_SERIAL_PORT_DEBUG_IDX)&& ((RxStatus & ULSR_FE)||(RxStatus & ULSR_OE)||(RxStatus & ULSR_PE)||(RxStatus & ULSR_RXFE)) )
      {
         DmConsolePrintf("Framing Errors = %d\r", m_FramingErrorCnt );
         DmConsolePrintf("Overrun Errors = %d\r", m_OverrunErrorCnt );
         DmConsolePrintf("Parity Errors = %d\r", m_ParityErrorCnt );
         DmConsolePrintf("Rx FIFO Errors = %d\r", m_FIFOErrorCnt );
      }

      if( RxStatus & (ULSR_OE|ULSR_RXFE))   // overrun error or rx FIFO
         m_pHwAccess->SetRegisterBits( UART_FCR_ADDR, ( UFCR_FCRFE | UFCR_RFR | UFCR_TFR ));       // reset FIFO

      isrFlags = UartIsr(m_HwCfg->bUart, 0, true, RxData, false);
      break;

   case UIIR_CTI:                  // Second priority, Receive character timeout interrupt code

     do
      {
         RxData   = m_pHwAccess->ReadRegister(UART_RHR_ADDR);//receive holding register address

         isrFlags = UartIsr( m_HwCfg->bUart, 0, true, RxData, false );

      }while(m_pHwAccess->ReadRegister(UART_LSR_ADDR)&ULSR_DR);
      break;


   case UIIR_THRE:                 // Third priority, Transmit buffer empty interrupt code

      FifoLevel = m_pHwAccess->ReadRegister(UART_TXLVL_ADDR);
      do
      {
         isrFlags = UartIsr(m_HwCfg->bUart, 0, rxReady,RxData, true);

         if(isrFlags & OS_ISR_FLAG_NO_MORE_DATA)
         {
            // Disable the USART Transmit interrupt
            m_pHwAccess->ClearRegisterBits( UART_IER_ADDR, UIER_THREIE);
            break;
         }
         else // still have something to send
         {
            // Write one byte to the transmit data register
            // data is in lower 16 bits of isr flags
            UART_SendData((UINT8)isrFlags);
         }

      }while(FifoLevel--);
      break;

   case UIIR_MSI:
   case UIIR_CTSRTS:
   default:                        //--- Unknown interrupt ---
      if( IsDmFieldDebugIndexActive(DM_FIELD_DM_SERIAL_PORT_DEBUG_IDX) )
      {
          DmConsolePrintf("Unkown Uart_SC16IS740 Interrupt\r");
      }

      break;
   }
}
/**
 * \author    Andrew Salmon
 * \brief     returns FCR registers content
 * \date      07/16/2009
 * \return    UINT8
 * \retval    FCR register content
 *
**/
UINT8 SC16IS740Hw::GetFCR_Register( void )
{
   return m_FCR;
}
/**
 * \author    Andrew Salmon
 * \brief     Sets bits in registers
 * \date      07/16/2009
 * \param     bReg register address
 * \param     bDataBits bits to set
 *
**/
void SC16IS740Hw::SetRegisterBits( UINT8 bReg, UINT8 bDataBits )
{
    UINT8 bData = 0;

    //Verify parameters
    if ( !IsValidRegister(bReg) )
       return;

    if( bReg != UART_FCR_ADDR )
    {
       //read the current register content
       bData = ReadRegister(bReg);
    }
    else // writing FCR (reg 02) or EFR (reg 02)
    {
        if(m_LCR != 0xBF) //writing FCR
        {
           bData = m_FCR;//current value
           m_FCR = (bData|bDataBits);//update value
        }
        else //0xBF if writing EFR instead of FCR
        {
           bData = m_EFR;//current value
           m_EFR = (bData|bDataBits);//update value
        }
    }

    //Write the byte
    WriteRegister(bReg, (bData | bDataBits));
}
/**
 * \author    Andrew Salmon
 * \brief     Clear bits in registers
 * \date      07/16/2009
 * \param     bReg register address
 * \param     bDataBits bits to set
 *
 **/
void SC16IS740Hw::ClearRegisterBits( UINT8 bReg, UINT8 bDataBits )
{
    UINT8 bData = 0;

    //Verify parameters
    if ( !IsValidRegister(bReg) )
        return;

    if( bReg != UART_FCR_ADDR )
    {
       //read the current register content
       bData = ReadRegister(bReg);
    }
    else // writing FCR (reg 02) or EFR (reg 02)
    {
        if(m_LCR != 0xBF) //writing FCR
        {
           bData = m_FCR;//current value
           m_FCR = (bData & ~bDataBits);//update value
        }
        else //0xBF if writing EFR instead of FCR
        {
           bData = m_EFR;//current value
           m_EFR = (bData & ~bDataBits);//update value
        }
    }

    //Write the byte
    WriteRegister(bReg, (bData & ~bDataBits));
}
/**
* \author    Andrew Salmon
* \brief     Constructor for SC16IS740 SPI hardware class
* \date      07/15/2009
* \param     pHwCfg - pointer to hardware configuration
*
**/
SC16IS740Hw_SPI::SC16IS740Hw_SPI( const SC16IS740Cfg *pHwCfg )
{
   //Save the variables
   m_Bus      = pHwCfg->bBus;
   m_BaseAddr = pHwCfg->bBaseAddr;
   m_FCR      = 0;
   m_LCR      = 0;
   m_EFR      = 0;
}
/**
* \author    Andrew Salmon
* \brief     read a SC16IS740 register on SPI
* \date      07/15/2009
* \param     reg - register to be read from
* \return    UINT8
* \retval    register value
**/
UINT8 SC16IS740Hw_SPI::ReadRegister(UINT8 reg)
{
  UINT8 bData[2];

  //Verify parameters
  if ( !IsValidRegister(reg) )
     return 0;

  // select the Host chip
  DmSPISelectDevice(m_Bus, m_BaseAddr, true);

  bData[0] = ((reg<<UART_REG_ADDR_BITNUM)|SPI_UART_READ_ADDR_MASK);
  bData[1] = 0;	    // dummy byte to clock out the register value
  // write to hardware
  DmSPIWriteBlock(m_Bus, m_BaseAddr, bData, 2);

  // this function will return the data received during the last write operation;
  DmSPIReadBlock(m_Bus, m_BaseAddr, bData, 2);
  // deselect the Host chip
  DmSPISelectDevice(m_Bus, m_BaseAddr, false);

  return bData[1];		// return the last FIFO byte
}
/**
* \author    Andrew Salmon
* \brief     write data to a SC16IS740 register on SPI
* \date      07/15/2009
* \param     reg - register to be written
* \param     value - dat to write
*
**/
void SC16IS740Hw_SPI::WriteRegister(UINT8 reg, UINT8 value)
{
  UINT8 bData[2];

  //Verify parameters
  if ( !IsValidRegister(reg) )
     return;

  if(reg == UART_LCR_ADDR)
  {
     m_LCR = value;
  }

  // select the Host chip
  DmSPISelectDevice(m_Bus, m_BaseAddr, true);

  bData[0] = (reg<<UART_REG_ADDR_BITNUM);//shift for address bit alignment
  bData[1] = value;

  // write to hardware
  DmSPIWriteBlock(m_Bus, m_BaseAddr, bData, 2);
  // deselect the Host chip
  DmSPISelectDevice(m_Bus, m_BaseAddr, false);

  return;
}
/**
* \author    Andrew Salmon
* \brief     Constructor for I2C hardware class
* \date      07/15/2009
* \param     pHwCfg - pointer to hardware configuration
*
**/
SC16IS740Hw_I2C::SC16IS740Hw_I2C( const SC16IS740Cfg *pHwCfg )
{
   //Save the variables
   m_Bus      = pHwCfg->bBus;
   m_BaseAddr = pHwCfg->bBaseAddr;
   m_FCR      = 0;
   m_LCR      = 0;
   m_EFR      = 0;
}
/**
* \author    Andrew Salmon
* \brief     read a SC16IS740 register I2C
* \date      07/15/2009
* \param     reg - register to be read from
* \return    UINT8
* \retval    register value
**/
UINT8 SC16IS740Hw_I2C::ReadRegister(UINT8 reg)
{
   return 0;		// return the last FIFO byte
}
/**
* \author    Andrew Salmon
* \brief     write data to a SC16IS740 register on I2C
* \date      07/15/2009
* \param     reg - register to be written
* \param     value - data to write
*
**/
void SC16IS740Hw_I2C::WriteRegister(UINT8 reg, UINT8 value)
{
   if(reg == UART_LCR_ADDR)
   {
      m_LCR = value;
   }
   return;
}
/**
 * \author      Andrew Salmon
 *
 * \date        09/17/2008
 *
 * \brief       Configure all programable parameters of UART
 *
 * \param       inst - which uart
 * \param       bps - baud rate
 * \param       stopBits - number of stop bits
 * \param       parity - aprity
 * \param       wlenght - word length
 * \param       hwcontrol - hardware flow control
 *
 */
void UartSC16IS740HwConfig(UINT32 inst, UINT32 bps, UINT32 stopBits, UINT32 parity, UINT32 wlenght, UINT32 hwcontrol)
{
   if(g_pExtUart)
      g_pExtUart->HwConfig(inst, bps, stopBits, parity, wlenght, hwcontrol);
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/15/2009
 *
 * \return      void
 *
 * \retval      void
 *
 * \brief       Change UART RTS state
 *
 * \param       state - Bit_SET (asserts)/Bit_Reset (deasserts)
 */
void UartSC16IS740RTS(BOOL state)
{
   if( g_pExtUart )
      g_pExtUart->AssertRTS(state);
}
/**
 * \author      Andrew Salmon
 *
 * \date        09/18/2008
 *
 * \return      BOOL
 *
 * \retval      state of /CTS bit in hardware register
 *
 * \brief       Reads state of UART CTS pin
 *
 */
BOOL UartSC16IS740GetCTS( void )
{
    if( g_pExtUart )
       return ( g_pExtUart->GetCTS());

    return false;
}
/**
 *  \author     Andrew Salmon
 *
 *  \date       12/20/08 11:09:AM
 *
 *  \brief      Post event to task queue
 *
 * \param       event event to queue
 */
void Uart_SC16IS740::PostEvent( void )
{
   UINT32 event;

    if(m_EventGroup)
    {
       OsQueue(m_EventGroup, (void * )&event, OS_WAIT_FOREVER );
    }
}
/**
 * \author    Andrew Salmon
 * \brief     This function handles the UART interrupt.
 * \date      07/14/2009
 * \param     param
 *
**/
void UartSC16IS740ExternalInterrupt( UINT32 param )
{
   if( g_pExtUart )
      g_pExtUart->PostEvent();
}
/**
 * \author      Andrew Salmon
 *
 * \date        07/16/2009
 *
 * \brief       Transmit data bytes
 *
 * \param       data pointer to serial data to transmit
 * \param       count number of bytes
 * \param       inst uart instance
 *
 */
void UartSC16IS740ConsoleTx(UINT8 *data, UINT32 count, UINT32 inst)
{
    if(g_pExtUart)
       g_pExtUart->ConsoleTx(data, count, inst);
}
/**
 * \author    Andrew Salmon
 * \brief     This function sends the first byte then init the tx interrupt to send the rest.
 * \date      07/14/2009
 * \param       uartNum uart instance
 * \param       data byte to transmit
**/
void UartSC16IS740HwUartTx(UINT32 uartNum, UINT8 data)
{
    if(g_pExtUart)
       g_pExtUart->HwUartTx(uartNum, data);
}
/**
 * \author      Pete McCormick
 *              Andrew Salmon
 *
 * \date        08/04/08
 *
 * \return      INT32
 *
 * \retval      0 success, no output written
 * \retval      1 success, raw data written to pData
 * \retval      2 success, ascii output written to pData
 * \retval      <0 failure
 *
 * \brief       Device driver entry point for this device
 *
 * \param       inst - which device if there are multiple of same type
 * \param       cmd - particular command
 * \param       pData - associated data pointer
 * \param       pDataBytes - associated bytes of data
 *
 * \note        There could be a queue in front of this if needed,
 *              to avoid race conditions.  Queue stuff would not go in this
 *              module.
 */
INT32 SC1740DeviceProcessInput(UINT32 inst, UINT32 cmd, UINT8 * pData, UINT32 * pDataBytes)
{
    //Verify the data
    if (*pDataBytes == 0)
        return 0;

    //Verify the #
    if ( !g_pExtUart )
    {
#ifdef DEBUG_SC16IS740_UART
        DmConsolePrintf("External UART class not initialized\r");
#endif
        return 0;
    }

    switch(cmd)
    {
        case DEVICE_PRINT:
         if ( IsDmFieldDebugIndexActive(DM_FIELD_DM_SERIAL_PORT_DEBUG_IDX) )
            //Print the debug information
            g_pExtUart->PrintDebug();
            break;

        case DEVICE_SET:
        case DEVICE_DETECT:
        case DEVICE_GET:
        case DEVICE_SELFTEST:
        case DEVICE_INIT:
          break;
        case DEVICE_TX:
        case DEVICE_RX:
          //Test tx/rx
          g_pExtUart->TestTxRx();
          break;

        case DEVICE_SET_DEBUG:
        case DEVICE_GET_DEBUG:
        default:
            break;
    }
    return 0;
}